-- 250801  1500
-- Ensure read committed isolation
SET TRANSACTION ISOLATION LEVEL READ COMMITTED;

-- Step 1: Copy data from PAC (etpproac) to FRQ (etpfetrq) and initialize Fetch Batch (FBT)

-- Log start
INSERT INTO public.etp_log (log_time, log_message)
VALUES (CURRENT_TIMESTAMP, 'FBatch1: Starting copy from etpproac to etpfetrq');

-- Declare variables for etpparam values
DO $$
DECLARE
    v_min_id TEXT;
    v_max_id TEXT;
    v_row_count BIGINT;
BEGIN
    -- Check etpparam row count
    SELECT COUNT(*) INTO v_row_count
    FROM public.etpparam
    WHERE oid = 1;

    -- Log etpparam status
    INSERT INTO public.etp_log (log_time, log_message)
    VALUES (CURRENT_TIMESTAMP, 
            'FBatch1: Found ' || v_row_count || ' etpparam row with oid=1');

    -- If no row with oid=1, log and exit
    IF v_row_count = 0 THEN
        INSERT INTO public.etp_log (log_time, log_message)
        VALUES (CURRENT_TIMESTAMP, 
                'FBatch1: No etpparam row with oid=1; skipping copy');
        RETURN;
    END IF;

    -- Get min_id and max_id from etpparam where oid=1
    SELECT TRIM(min_id::TEXT), TRIM(max_id::TEXT)
    INTO v_min_id, v_max_id
    FROM public.etpparam
    WHERE oid = 1;

    -- Log selected etpparam values
    INSERT INTO public.etp_log (log_time, log_message)
    VALUES (CURRENT_TIMESTAMP, 
            'FBatch1: Using etpparam: min_id=' || v_min_id || ', max_id=' || v_max_id);

    -- Log etpproac raw importid values
    INSERT INTO public.etp_log (log_time, log_message)
    SELECT CURRENT_TIMESTAMP, 
           'FBatch1: Raw etpproac: ImportID="' || importid || 
           '", Length=' || LENGTH(importid) || 
           ', Trimmed="' || TRIM(importid) || 
           '", OID=' || oid
    FROM public.etpproac
    WHERE TRIM(importid) BETWEEN v_min_id AND v_max_id;

    -- Clear the FRQ table
    TRUNCATE TABLE public.etpfetrq;

    -- Log TRUNCATE
    INSERT INTO public.etp_log (log_time, log_message)
    VALUES (CURRENT_TIMESTAMP, 'FBatch1: Cleared etpfetrq table');

    -- Log etpproac records after filtering
    INSERT INTO public.etp_log (log_time, log_message)
    SELECT CURRENT_TIMESTAMP, 
           'FBatch1: Filtered etpproac: ImportID="' || importid || 
           '", Trimmed="' || TRIM(importid) || 
           '", OID=' || oid || 
           ', TemplateNumber=' || COALESCE(templatenumber::TEXT, 'NULL') || 
           ', DBTableName=' || COALESCE(dbtablename, 'NULL') || 
           ', UserName=' || COALESCE(username, 'NULL') || 
           ', ACode=' || COALESCE(acode, 'NULL')
    FROM public.etpproac
    WHERE TRIM(importid) BETWEEN v_min_id AND v_max_id;

    -- Insert data from PAC (etpproac) into FRQ (etpfetrq), selecting one record per ImportID
    INSERT INTO public.etpfetrq (oid, templatenumber, dbtablename, username, importid, acode)
    SELECT DISTINCT ON (TRIM(importid)) 
           oid, templatenumber, dbtablename, username, importid, acode
    FROM public.etpproac
    WHERE TRIM(importid) BETWEEN v_min_id AND v_max_id
    ORDER BY TRIM(importid), oid DESC;

    -- Log FRQ insertion
    INSERT INTO public.etp_log (log_time, log_message)
    SELECT CURRENT_TIMESTAMP, 
           'FBatch1: Copied ' || COUNT(*) || ' records to etpfetrq: ' ||
           COALESCE(STRING_AGG('ImportID="' || importid || '", OID=' || oid, '; '), 'No records inserted')
    FROM public.etpfetrq;
END $$;

-- Initialize Fetch Batch in FBT
DO $$ 
DECLARE
    current_batch_oid BIGINT;
    start_time TEXT := TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS');
    batchnumber BIGINT := COALESCE((SELECT MAX(BatchNumber) FROM public.etpfetbt), 0) + 1;
    templatenumber BIGINT := (SELECT TemplateNumber FROM public.ETPProHd);
    username TEXT := (SELECT UserName FROM public.ETPProHd);
    dbtablename TEXT := (SELECT DBTableName FROM public.ETPProHd);
    recordcount BIGINT := (SELECT COUNT(*) FROM public.etpfetrq);
    status TEXT := 'Pending';
BEGIN
    INSERT INTO public.etpfetbt (
        BatchNumber, TemplateNumber, UserName, DBTableName, RecordCount, StartTime, Status
    ) VALUES (
        batchnumber, templatenumber, username, dbtablename, recordcount, start_time, status
    )
    RETURNING OID INTO current_batch_oid;

    INSERT INTO public.etp_log (log_time, log_message)
    VALUES (CURRENT_TIMESTAMP, 
            'FBatch1: Created batch ' || batchnumber || 
            ', OID=' || current_batch_oid || 
            ', RecordCount=' || recordcount);

    INSERT INTO public.etp_log (log_time, log_message)
    VALUES (CURRENT_TIMESTAMP, 'FBatch1: Completed - Data copied from PAC to FRQ and batch initialized in FBT');
END $$;
 