-- FBatch5.sql
-- Purpose: Waits for etpfetrt to be populated by ETPFet.exe /SQL, updates etpfetbt and etppaclog, logs results
-- Version: v250805b

DO $$
DECLARE
    rt_count BIGINT;
    retry_count INTEGER := 0;
    max_retries INTEGER := 60; -- 5 minutes at 5-second intervals
    retry_delay INTEGER := 5; -- seconds
    log_message TEXT;
BEGIN
    -- Verify etpfetrt accessibility
    IF NOT EXISTS (
        SELECT FROM information_schema.tables 
        WHERE table_schema = 'public' AND table_name = 'etpfetrt'
    ) THEN
        log_message := 'FBatch5: etpfetrt table does not exist.';
        INSERT INTO public.etp_log (log_time, log_message)
        VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE EXCEPTION '%', log_message;
    END IF;

    -- Log table found
    INSERT INTO public.etp_log (log_time, log_message)
    VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), 'FBatch5: etpfetrt table found, starting poll');

    -- Loop to check etpfetrt
    LOOP
        retry_count := retry_count + 1;
        BEGIN
            SELECT COUNT(*) INTO rt_count FROM public.etpfetrt;
        EXCEPTION
            WHEN OTHERS THEN
                log_message := format('FBatch5: Error querying etpfetrt: %s', SQLERRM);
                INSERT INTO public.etp_log (log_time, log_message)
                VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
                RAISE EXCEPTION '%', log_message;
        END;
        log_message := format('FBatch5: Retry %s/%s: etpfetrt has %s rows', retry_count, max_retries, rt_count);
        INSERT INTO public.etp_log (log_time, log_message)
        VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
        IF rt_count > 0 THEN
            log_message := format('FBatch5: etpfetrt contains %s rows. Processing complete.', rt_count);
            INSERT INTO public.etp_log (log_time, log_message)
            VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
            -- Embedded FBatch7.sql logic
            DECLARE
                current_batch_oid BIGINT;
                current_batch_number BIGINT;
                end_time TIMESTAMP := CURRENT_TIMESTAMP;
                start_time VARCHAR(30);
                execution_time TEXT;
                paclog_count BIGINT;
            BEGIN
                -- Get the latest Pending batch
                SELECT DISTINCT ON (Status) oid, BatchNumber INTO current_batch_oid, current_batch_number
                FROM public.etpfetbt
                WHERE Status = 'Pending'
                ORDER BY Status, oid DESC;
                IF current_batch_oid IS NULL THEN
                    log_message := 'FBatch5: No Pending batch found in etpfetbt.';
                    INSERT INTO public.etp_log (log_time, log_message)
                    VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
                ELSE
                    SELECT StartTime INTO start_time FROM public.etpfetbt WHERE oid = current_batch_oid;
                    IF start_time IS NULL THEN
                        log_message := format('FBatch5: Start time not found for batch OID: %s', current_batch_oid);
                        INSERT INTO public.etp_log (log_time, log_message)
                        VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
                    ELSE
                        -- Update etpfetbt
                        execution_time := TO_CHAR(end_time - start_time::TIMESTAMP, 'HH24:MI:SS');
                        UPDATE public.etpfetbt
                        SET EndTime = TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'),
                            ExecutionTime = execution_time,
                            Status = 'Completed'
                        WHERE oid = current_batch_oid;
                        log_message := format('FBatch5: Batch OID %s marked as Completed. EndTime: %s, ExecutionTime: %s',
                                             current_batch_oid, TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'), execution_time);
                        INSERT INTO public.etp_log (log_time, log_message)
                        VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
                        -- Update etppaclog to Consumed
                        UPDATE public.etppaclog
                        SET status = 'Consumed'
                        WHERE batchnumber = current_batch_number AND status = 'Pending';
                        GET DIAGNOSTICS paclog_count = ROW_COUNT;
                        log_message := format('FBatch5: Updated %s etppaclog records to Consumed for batchnumber: %s',
                                             paclog_count, current_batch_number);
                        INSERT INTO public.etp_log (log_time, log_message)
                        VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
                    END IF;
                END IF;
            EXCEPTION
                WHEN OTHERS THEN
                    log_message := format('FBatch5: Error updating etpfetbt or etppaclog: %s', SQLERRM);
                    INSERT INTO public.etp_log (log_time, log_message)
                    VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
            END;
            EXIT;
        END IF;
        IF retry_count >= max_retries THEN
            log_message := format('FBatch5: Max retries (%s) reached. etpfetrt still empty.', max_retries);
            INSERT INTO public.etp_log (log_time, log_message)
            VALUES (TO_CHAR(CURRENT_TIMESTAMP, 'YYYY-MM-DD HH24:MI:SS'), log_message);
            RAISE EXCEPTION '%', log_message;
        END IF;
        PERFORM pg_sleep(retry_delay); -- Delay in seconds
    END LOOP;
END $$;