/* Set schema to public */
SET search_path TO public;

/* Drop function to clear any cached or conflicting definition */
DROP FUNCTION IF EXISTS public.sendatrix_audit_triggers() CASCADE;

DO $$ BEGIN
    RAISE NOTICE 'Dropped public.sendatrix_audit_triggers at %', to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
END $$;

/* Create table with mixed-case column names to match previous working state */
CREATE TABLE IF NOT EXISTS public.sendatrix_audit_log (
    AuditId integer PRIMARY KEY,
    audittime varchar(20),
    AuditType varchar(10) CHECK (AuditType IN ('Trigger', 'Listener', 'QueryLog', 'Extension', 'SSL')),
    Status varchar(20) CHECK (Status IN ('Authorized', 'Unauthorized')),
    Details varchar(100)
);

DO $$ BEGIN
    RAISE NOTICE 'Table public.sendatrix_audit_log created or exists at %', to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
END $$;

/* Ensure sequence exists */
CREATE SEQUENCE IF NOT EXISTS public.sendatrix_audit_log_AuditId_seq
    OWNED BY public.sendatrix_audit_log.AuditId;

/* Set defaults only if not already set */
DO $$ BEGIN
    IF NOT EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'public' AND table_name = 'sendatrix_audit_log'
        AND column_name = 'audittime' AND column_default IS NOT NULL
    ) THEN
        ALTER TABLE public.sendatrix_audit_log
            ALTER COLUMN audittime SET DEFAULT to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
        RAISE NOTICE 'Default set for audittime at %', to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
    END IF;
    IF NOT EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'public' AND table_name = 'sendatrix_audit_log'
        AND column_name = 'AuditId' AND column_default IS NOT NULL
    ) THEN
        ALTER TABLE public.sendatrix_audit_log
            ALTER COLUMN AuditId SET DEFAULT nextval('public.sendatrix_audit_log_AuditId_seq');
        RAISE NOTICE 'Default set for AuditId at %', to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
    END IF;
END $$;

GRANT SELECT, INSERT, UPDATE, DELETE ON public.sendatrix_audit_log TO postgres;
GRANT USAGE, SELECT ON SEQUENCE public.sendatrix_audit_log_AuditId_seq TO postgres;

CREATE TABLE IF NOT EXISTS public.sendatrix_trigger_whitelist (
    oid bigserial PRIMARY KEY,
    TriggerName text NOT NULL,
    TableName text NOT NULL,
    CreatedBy text NOT NULL,
    Description text,
    CONSTRAINT tawtlst_unique UNIQUE (TriggerName, TableName)
);

DO $$ BEGIN
    RAISE NOTICE 'Table public.sendatrix_trigger_whitelist created or exists at %', to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
END $$;

GRANT SELECT, INSERT, UPDATE, DELETE ON public.sendatrix_trigger_whitelist TO postgres;
GRANT USAGE, SELECT ON SEQUENCE public.sendatrix_trigger_whitelist_oid_seq TO postgres;

INSERT INTO public.sendatrix_trigger_whitelist (TriggerName, TableName, CreatedBy, Description)
SELECT 'insert_prq_trigger', LOWER(DBTablename), 'postgres', 'Insert trigger for production requests'
FROM public.etpprohd
WHERE DBTablename IS NOT NULL
LIMIT 1
ON CONFLICT ON CONSTRAINT tawtlst_unique DO NOTHING;

CREATE TABLE IF NOT EXISTS public.sendatrix_protected_tables (
    oid bigserial PRIMARY KEY,
    DBTblNme text NOT NULL,
    Description text,
    CONSTRAINT taprodbt_unique UNIQUE (DBTblNme)
);

DO $$ BEGIN
    RAISE NOTICE 'Table public.sendatrix_protected_tables created or exists at %', to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
END $$;

GRANT SELECT, INSERT, UPDATE, DELETE ON public.sendatrix_protected_tables TO postgres;
GRANT USAGE, SELECT ON SEQUENCE public.sendatrix_protected_tables_oid_seq TO postgres;

INSERT INTO public.sendatrix_protected_tables (DBTblNme, Description)
SELECT LOWER(DBTablename), 'Dynamic protected table from etpprohd'
FROM public.etpprohd
WHERE DBTablename IS NOT NULL
LIMIT 1
ON CONFLICT ON CONSTRAINT taprodbt_unique DO NOTHING;

INSERT INTO public.sendatrix_protected_tables (DBTblNme, Description)
VALUES ('etpprorq', 'MBTIP request queuing table')
ON CONFLICT ON CONSTRAINT taprodbt_unique DO NOTHING;

INSERT INTO public.sendatrix_protected_tables (DBTblNme, Description)
VALUES ('etpprobf', 'MBTIP buffer table for production requests')
ON CONFLICT ON CONSTRAINT taprodbt_unique DO NOTHING;

CREATE OR REPLACE FUNCTION public.sendatrix_audit_triggers()
RETURNS TABLE (
    triggername text,
    audited_tablename text,
    event_type text,
    trigger_function text,
    CreatedBy text,
    status text,
    details text
) AS $$
DECLARE
    dynamic_table_name text;
    v_computed_status text;
    unauthorized_count integer;
    trigger_was_enabled boolean;
BEGIN
    RAISE NOTICE 'Executing public.sendatrix_audit_triggers at %', to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
    SELECT LOWER(DBTablename) INTO dynamic_table_name FROM public.etpprohd WHERE DBTablename IS NOT NULL LIMIT 1;
    IF dynamic_table_name IS NULL THEN
        RAISE NOTICE 'No dynamic table found in etpprohd';
    ELSE
        RAISE NOTICE 'Dynamic Table Name: %', dynamic_table_name;
    END IF;
    SELECT EXISTS (
        SELECT 1 FROM public.ETPConfig
        WHERE option_name = 'TriggerEnabled' AND LOWER(option_value) = 'true'
    ) INTO trigger_was_enabled;
    SELECT COUNT(*) INTO unauthorized_count
    FROM information_schema.triggers t
    LEFT JOIN public.sendatrix_trigger_whitelist w 
        ON t.trigger_name::text = w.TriggerName AND LOWER(t.event_object_table::text) = w.TableName
    WHERE LOWER(t.event_object_table::text) IN (COALESCE(dynamic_table_name, ''), 'etpprorq', 'etpprobf')
    AND (LOWER(t.trigger_name) != 'insert_prq_trigger' OR LOWER(t.event_object_table) != COALESCE(dynamic_table_name, ''))
    AND w.TriggerName IS NULL;
    IF unauthorized_count > 0 THEN
        UPDATE public.ETPConfig
        SET option_value = 'false'
        WHERE option_name = 'TriggerEnabled';
        RAISE NOTICE 'Rogue trigger detected. TriggerEnabled set to false.';
    END IF;
    FOR v_computed_status IN (
        SELECT t.trigger_name::text || ' on ' || t.event_object_table::text || ' in schema ' || t.trigger_schema::text
        FROM information_schema.triggers t
        WHERE LOWER(t.event_object_table::text) IN (COALESCE(dynamic_table_name, ''), 'etpprorq', 'etpprobf')
    ) LOOP
        RAISE NOTICE 'Trigger, Table, and Schema: %', v_computed_status;
    END LOOP;
    INSERT INTO public.sendatrix_audit_log (
        AuditId, audittime, AuditType, Status, Details
    )
    SELECT 
        nextval('public.sendatrix_audit_log_AuditId_seq'),
        to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS'),
        'Trigger',
        LEFT(
            CASE 
                WHEN LOWER(t.trigger_name) = 'insert_prq_trigger' AND LOWER(t.event_object_table) = COALESCE(dynamic_table_name, '') THEN 'Authorized'
                WHEN w.TriggerName IS NOT NULL THEN 'Authorized'
                ELSE 'Unauthorized'
            END,
            20
        ) AS insert_status,
        LEFT(
            CASE 
                WHEN LOWER(t.trigger_name) = 'insert_prq_trigger' AND LOWER(t.event_object_table) = COALESCE(dynamic_table_name, '') THEN 'Authorized MBTIP trigger'
                WHEN w.TriggerName IS NOT NULL THEN 'Authorized MBTIP trigger'
                ELSE 
                    'Rogue trigger detected' || 
                    CASE 
                        WHEN trigger_was_enabled AND unauthorized_count > 0 THEN ': Trigger mode disabled'
                        ELSE ''
                    END || 
                    '. Trig: ' || COALESCE(t.trigger_name::text, 'unknown') || 
                    ', Tbl: ' || COALESCE(t.event_object_table::text, 'unknown') || 
                    ', Fn: ' || COALESCE(p.proname::text, 'unknown') || 
                    ', Own: ' || COALESCE(r.rolname::text, 'unknown') || 
                    ', Ev: ' || COALESCE(t.event_manipulation::text, 'unknown')
            END,
            100
        ) AS Details
    FROM information_schema.triggers t
    JOIN pg_trigger pt ON t.trigger_name::text = pt.tgname::text
    JOIN pg_proc p ON pt.tgfoid = p.oid
    JOIN pg_roles r ON p.proowner = r.oid
    LEFT JOIN public.sendatrix_trigger_whitelist w 
        ON t.trigger_name::text = w.TriggerName AND LOWER(t.event_object_table::text) = w.TableName
    WHERE LOWER(t.event_object_table::text) IN (COALESCE(dynamic_table_name, ''), 'etpprorq', 'etpprobf');
    SELECT 
        LEFT(
            CASE 
                WHEN LOWER(t.trigger_name) = 'insert_prq_trigger' AND LOWER(t.event_object_table) = COALESCE(dynamic_table_name, '') THEN 'Authorized'
                WHEN w.TriggerName IS NOT NULL THEN 'Authorized'
                ELSE 'Unauthorized'
            END,
            20
        ) INTO v_computed_status
    FROM information_schema.triggers t
    LEFT JOIN public.sendatrix_trigger_whitelist w 
        ON t.trigger_name::text = w.TriggerName AND LOWER(t.event_object_table::text) = w.TableName
    WHERE LOWER(t.event_object_table::text) IN (COALESCE(dynamic_table_name, ''), 'etpprorq', 'etpprobf')
    ORDER BY t.trigger_name DESC
    LIMIT 1;
    RAISE NOTICE 'Computed Status: %', v_computed_status;
    RETURN QUERY
    SELECT 
        t.trigger_name::text AS triggername,
        t.event_object_table::text AS audited_tablename,
        t.event_manipulation::text AS event_type,
        p.proname::text AS trigger_function,
        r.rolname::text AS CreatedBy,
        LEFT(
            CASE 
                WHEN LOWER(t.trigger_name) = 'insert_prq_trigger' AND LOWER(t.event_object_table) = COALESCE(dynamic_table_name, '') THEN 'Authorized'
                WHEN w.TriggerName IS NOT NULL THEN 'Authorized'
                ELSE 'Unauthorized'
            END,
            20
        ) AS status,
        LEFT(
            CASE 
                WHEN LOWER(t.trigger_name) = 'insert_prq_trigger' AND LOWER(t.event_object_table) = COALESCE(dynamic_table_name, '') THEN 'Authorized MBTIP trigger'
                WHEN w.TriggerName IS NOT NULL THEN 'Authorized MBTIP trigger'
                ELSE 
                    'Rogue trigger detected' || 
                    CASE 
                        WHEN trigger_was_enabled AND unauthorized_count > 0 THEN ': Trigger mode disabled'
                        ELSE ''
                    END || 
                    '. Trig: ' || COALESCE(t.trigger_name::text, 'unknown') || 
                    ', Tbl: ' || COALESCE(t.event_object_table::text, 'unknown') || 
                    ', Fn: ' || COALESCE(p.proname::text, 'unknown') || 
                    ', Own: ' || COALESCE(r.rolname::text, 'unknown') || 
                    ', Ev: ' || COALESCE(t.event_manipulation::text, 'unknown')
            END,
            100
        ) AS details
    FROM information_schema.triggers t
    JOIN pg_trigger pt ON t.trigger_name::text = pt.tgname::text
    JOIN pg_proc p ON pt.tgfoid = p.oid
    JOIN pg_roles r ON p.proowner = r.oid
    LEFT JOIN public.sendatrix_trigger_whitelist w 
        ON t.trigger_name::text = w.TriggerName AND LOWER(t.event_object_table::text) = w.TableName
    WHERE LOWER(t.event_object_table::text) IN (COALESCE(dynamic_table_name, ''), 'etpprorq', 'etpprobf');
END;
$$ LANGUAGE plpgsql SECURITY DEFINER;

DO $$ BEGIN
    RAISE NOTICE 'Function public.sendatrix_audit_triggers created at %', to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
END $$;

GRANT EXECUTE ON FUNCTION public.sendatrix_audit_triggers() TO postgres;

CREATE OR REPLACE FUNCTION public.sendatrix_audit_triggers_summary()
RETURNS text AS $$
DECLARE
    result text;
    unauthorized_count integer;
    trigger_was_enabled boolean;
BEGIN
    RAISE NOTICE 'Executing public.sendatrix_audit_triggers_summary at %', to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
    SELECT COUNT(*) INTO unauthorized_count
    FROM public.sendatrix_audit_log
    WHERE Status = 'Unauthorized' AND audittime = to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
    SELECT EXISTS (
        SELECT 1 FROM public.ETPConfig
        WHERE option_name = 'TriggerEnabled' AND LOWER(option_value) = 'true'
    ) INTO trigger_was_enabled;
    IF unauthorized_count > 0 THEN
        SELECT 
            'Rogue trigger detected' || 
            CASE 
                WHEN trigger_was_enabled THEN ': Trigger mode disabled'
                ELSE ''
            END || 
            '. Trig: ' || COALESCE(triggername, 'unknown') || 
            ', Tbl: ' || COALESCE(audited_tablename, 'unknown')
        INTO result
        FROM public.sendatrix_audit_triggers()
        WHERE status = 'Unauthorized'
        LIMIT 1;
    ELSE
        result := 'All triggers authorized';
    END IF;
    RETURN result;
END;
$$ LANGUAGE plpgsql;

GRANT EXECUTE ON FUNCTION public.sendatrix_audit_triggers_summary() TO postgres;

DROP FUNCTION IF EXISTS public.sendatrix_disable_trigger_on_rogue();

CREATE OR REPLACE FUNCTION public.sendatrix_disable_trigger_on_rogue()
RETURNS void AS $$
DECLARE
    unauthorized_count integer;
BEGIN
    RAISE NOTICE 'Executing public.sendatrix_disable_trigger_on_rogue at %', to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
    SELECT COUNT(*) INTO unauthorized_count
    FROM public.sendatrix_audit_log
    WHERE Status = 'Unauthorized' AND audittime = to_char(current_timestamp AT TIME ZONE 'America/New_York', 'YYYY-MM-DD HH24:MI:SS');
    IF unauthorized_count > 0 THEN
        UPDATE public.ETPConfig
        SET option_value = 'false'
        WHERE option_name = 'TriggerEnabled';
        RAISE NOTICE 'Rogue trigger detected. TriggerEnabled set to false.';
    END IF;
END;
$$ LANGUAGE plpgsql;

GRANT EXECUTE ON FUNCTION public.sendatrix_disable_trigger_on_rogue() TO postgres;
