-- CreateTriggerInsertPRQFunction_Auto.sql v250702a
-- Buffered PRQ Trigger Insert Function for CTP/ETP
-- Prevents duplicate batch creation and writes to ETPProBf
-- Removed BufferMode check to standardize on buffered method

CREATE OR REPLACE FUNCTION trigger_insert_prq()
RETURNS TRIGGER AS $$
DECLARE
    templatenumber_local BIGINT;
    username_var TEXT;
    dbtablename_var TEXT;
    unique_column TEXT;
    field_rec RECORD;
    data_value TEXT;
    source_id BIGINT;
    import_id TEXT;
BEGIN
    -- Retrieve header configuration
    SELECT TemplateNumber, UserName, DBTableName, UniqueSourceColumn
    INTO templatenumber_local, username_var, dbtablename_var, unique_column
    FROM ETPProHd
    LIMIT 1;

    -- Exit if not the intended table
    IF TG_TABLE_NAME != dbtablename_var THEN
        RETURN NEW;
    END IF;

    -- Exit if trigger mode is disabled
    IF NOT EXISTS (
        SELECT 1 FROM ETPConfig
        WHERE option_name = 'TriggerEnabled' AND LOWER(option_value) = 'true'
    ) THEN
        RETURN NEW;
    END IF;

    -- Get SourceID value from the NEW record (becomes ImportID)
    EXECUTE format('SELECT ($1).%I::BIGINT', unique_column)
    USING NEW INTO source_id;

    import_id := source_id::TEXT;

    -- Insert each mapped field into the buffer table (ETPProBf)
    FOR field_rec IN
        SELECT DataFieldName, DataFieldNumber
        FROM ETPProFM
        WHERE TemplateNumber = templatenumber_local
    LOOP
        EXECUTE format('SELECT ($1).%I::TEXT', field_rec.DataFieldName)
        USING NEW INTO data_value;

        INSERT INTO ETPProBf (
            guestbook,
            location,
            templatenumber,
            username,
            dbtablename,
            importid,
            datafieldnumber,
            datafieldname,
            datavalue
        ) VALUES (
            NULL,
            (SELECT Location FROM ETPProHd WHERE DBTableName = TG_TABLE_NAME LIMIT 1),
            templatenumber_local,
            username_var,
            dbtablename_var,
            import_id,
            field_rec.DataFieldNumber,
            field_rec.DataFieldName,
            data_value
        );
    END LOOP;

    -- Only insert a new batch record if no "Ready" batch currently exists for this table
    IF NOT EXISTS (
        SELECT 1 FROM ETPProBt
        WHERE status = 'Ready' AND dbtablename = dbtablename_var
    ) THEN
        INSERT INTO ETPProBt (
            BatchNumber,
            TemplateNumber,
            UserName,
            DBTableName,
            RecordCount,
            StartTime,
            Status
        ) VALUES (
            (SELECT COALESCE(MAX(BatchNumber), 0) + 1 FROM ETPProBt),
            templatenumber_local,
            username_var,
            dbtablename_var,
            1,
            CURRENT_TIMESTAMP,
            'Ready'
        );
    END IF;

    RETURN NEW;
END $$ LANGUAGE plpgsql;