-- FlushBufferedToPRQ.sql (v250513a)
-- Move records from ETPProBf (PBF) to ETPProRq (PRQ), assigning new OIDs
-- Cleans PRQ before insert to avoid leftover Dx field errors

DO $$
DECLARE
    rec RECORD;
    new_oid BIGINT := 1;
    flushed_count INT := 0;
    actual_recordcount BIGINT := 0;
    templatenumber_local BIGINT;
    username_var TEXT;
    dbtablename_var TEXT;
    batch_oid BIGINT;
    start_time TIMESTAMP := CURRENT_TIMESTAMP;
BEGIN
    -- Clean PRQ table if any buffered records exist
    IF EXISTS (SELECT 1 FROM etpprobf) THEN
        TRUNCATE TABLE etpproRq;
        RAISE NOTICE 'PRQ table truncated before flush.';
    END IF;

    -- Get values from header
    SELECT TemplateNumber, UserName, DBTableName
    INTO templatenumber_local, username_var, dbtablename_var
    FROM ETPProHd
    LIMIT 1;

    -- Flush records from buffer to PRQ
    FOR rec IN SELECT * FROM etpproBf ORDER BY importid, datafieldnumber LOOP
        INSERT INTO etpproRq (
            oid,
            guestbook,
            location,
            templatenumber,
            username,
            dbtablename,
            importid,
            datafieldnumber,
            datafieldname,
            datavalue
        ) VALUES (
            new_oid,
            rec.guestbook,
            rec.location,
            rec.templatenumber,
            rec.username,
            rec.dbtablename,
            rec.importid,
            rec.datafieldnumber,
            rec.datafieldname,
            rec.datavalue
        );

        new_oid := new_oid + 1;
        flushed_count := flushed_count + 1;
    END LOOP;

    RAISE NOTICE 'Flushed % records from ETPProBf to ETPProRq.', flushed_count;

    -- Calculate actual ImportID count
    SELECT COUNT(DISTINCT importid) INTO actual_recordcount FROM etpproRq;

    -- Insert into ETPProBt and capture OID
    INSERT INTO etpprobt (
        BatchNumber,
        TemplateNumber,
        UserName,
        DBTableName,
        RecordCount,
        StartTime,
        EndTime,
        ExecutionTime,
        Status
    ) VALUES (
        (SELECT COALESCE(MAX(BatchNumber), 0) + 1 FROM etpprobt),
        templatenumber_local,
        username_var,
        dbtablename_var,
        actual_recordcount,
        start_time,
        CURRENT_TIMESTAMP,
        TO_CHAR(CURRENT_TIMESTAMP - start_time, 'HH24:MI:SS'),
        'Ready'
    ) RETURNING oid INTO batch_oid;

    -- Clear buffer if flushed
    IF flushed_count > 0 THEN
        DELETE FROM etpproBf;
        RAISE NOTICE 'ETPProBf (buffer) has been cleared.';
    ELSE
        RAISE NOTICE 'No buffered records to flush.';
    END IF;
END $$;
