-- PBatch1.sql
-- Purpose: Populate ETPProRq for batch mode, log to etp_log with LogOID
-- Version: TriggerBuffered_v250715c, 2025-07-15
DO $$
DECLARE
    trigger_enabled BOOLEAN := (SELECT option_value::BOOLEAN FROM etpconfig WHERE option_name = 'TriggerEnabled' LIMIT 1);
    start_time TIMESTAMP := NOW();
    log_message TEXT;
BEGIN
    -- Log start of truncate
    log_message := format('PBatch1.sql: Starting conditional truncate of ETPProRq. StartTime: %s', 
                         TO_CHAR(start_time, 'YYYY-MM-DD HH24:MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    IF NOT trigger_enabled THEN
        TRUNCATE TABLE etpprorq;
        log_message := format('PBatch1.sql: Truncated ETPProRq. Elapsed: %s', 
                             TO_CHAR(NOW() - start_time, 'MI:SS'));
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE NOTICE '%', log_message;
    ELSE
        log_message := format('PBatch1.sql: Trigger mode enabled, skipped truncate. Elapsed: %s', 
                             TO_CHAR(NOW() - start_time, 'MI:SS'));
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE NOTICE '%', log_message;
    END IF;
END $$;

-- Step 2: Insert Protection Requests into ETPProRq Table (Batch Mode Only)
DO $$
DECLARE
    current_batch_oid BIGINT;
    start_time TIMESTAMP := NOW();
    new_batchnumber BIGINT := COALESCE((SELECT MAX(batchnumber) FROM etpprobt), 0) + 1;
    templatenumber_local BIGINT;
    username_var TEXT;
    dbtablename_local TEXT;
    record_count_local BIGINT := 0;
    status TEXT := 'Pending';
    source_rec RECORD;
    field_rec RECORD;
    data_value TEXT;
    min_id BIGINT := (SELECT COALESCE(min_id, 0) FROM ETPParam);
    max_id BIGINT := (SELECT COALESCE(max_id, 0) FROM ETPParam);
    oid_counter BIGINT := 1;
    unique_id_column TEXT;
    trigger_enabled BOOLEAN := (SELECT option_value::BOOLEAN FROM etpconfig WHERE option_name = 'TriggerEnabled' LIMIT 1);
    log_message TEXT;
BEGIN
    -- Log start of population
    log_message := format('PBatch1.sql: Starting ETPProRq population. StartTime: %s', 
                         TO_CHAR(start_time, 'YYYY-MM-DD HH24:MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    IF trigger_enabled THEN
        log_message := format('PBatch1.sql: Trigger mode enabled, skipping batch-mode PRQ build. Elapsed: %s', 
                             TO_CHAR(NOW() - start_time, 'MI:SS'));
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE NOTICE 'Trigger mode is enabled; skipping batch-mode PRQ build.';
        RETURN;
    END IF;

    -- Get header config
    SELECT TemplateNumber, UserName, DBTableName, UniqueSourceColumn
    INTO templatenumber_local, username_var, dbtablename_local, unique_id_column
    FROM ETPProHd
    LIMIT 1;

    -- Log header config
    log_message := format('PBatch1.sql: Using TemplateNumber: %s, DBTableName: %s, UniqueSourceColumn: %s. Elapsed: %s', 
                         templatenumber_local, dbtablename_local, unique_id_column, TO_CHAR(NOW() - start_time, 'MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- Create batch header
    INSERT INTO etpprobt (
        batchnumber, templatenumber, username, dbtablename, recordcount, starttime, status
    ) VALUES (
        new_batchnumber, templatenumber_local, username_var, dbtablename_local, 0, TO_CHAR(start_time, 'YYYY-MM-DD HH24:MI:SS'), status
    ) RETURNING batchnumber INTO current_batch_oid;

    -- Log batch header creation
    log_message := format('PBatch1.sql: Batch record created with BatchNumber: %s. Elapsed: %s', 
                         new_batchnumber, TO_CHAR(NOW() - start_time, 'MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE 'Batch record created with BatchNumber: %, StartTime: %s', new_batchnumber, TO_CHAR(start_time, 'YYYY-MM-DD HH24:MI:SS');

    -- Update parameter table with current batch oid
    UPDATE ETPParam
    SET batch_oid = current_batch_oid
    WHERE oid = 1;

    -- Log parameter update
    log_message := format('PBatch1.sql: Updated ETPParam with Batch OID: %s. Elapsed: %s', 
                         current_batch_oid, TO_CHAR(NOW() - start_time, 'MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE 'ETPParam updated with Batch OID: %', current_batch_oid;

    -- Loop through source table directly
    FOR source_rec IN EXECUTE format(
        'SELECT %1$I AS importid, * FROM %2$I WHERE (%1$I BETWEEN %3$L AND %4$L) OR (%3$L = 0 AND %4$L = 0) ORDER BY %1$I',
        unique_id_column, dbtablename_local, min_id, max_id
    )
    LOOP
        FOR field_rec IN
            SELECT DataFieldName, DataFieldNumber FROM ETPProFM
            WHERE TemplateNumber = templatenumber_local
        LOOP
            EXECUTE format(
                'SELECT %I FROM %I WHERE %I = %L',
                field_rec.DataFieldName, dbtablename_local, unique_id_column, source_rec.importid
            )
            INTO data_value;

            INSERT INTO ETPProRq (
                OID, GuestBook, Location, TemplateNumber, UserName, DBTableName, ImportID, DataFieldNumber, DataFieldName, DataValue
            ) VALUES (
                oid_counter, NULL,
                (SELECT Location FROM ETPProHd WHERE DBTableName = dbtablename_local LIMIT 1),
                templatenumber_local,
                username_var,
                dbtablename_local,
                source_rec.importid,
                field_rec.DataFieldNumber,
                field_rec.DataFieldName,
                data_value
            );

            oid_counter := oid_counter + 1;
        END LOOP;

        record_count_local := record_count_local + 1;
    END LOOP;

    -- Update record count in batch header
    UPDATE etpprobt
    SET recordcount = record_count_local
    WHERE batchnumber = current_batch_oid;

    -- Log record count update
    log_message := format('PBatch1.sql: Updated etpprobt with %s records for BatchNumber: %s. Elapsed: %s', 
                         record_count_local, new_batchnumber, TO_CHAR(NOW() - start_time, 'MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE 'Updated etpprobt with % records for BatchNumber: %', record_count_local, new_batchnumber;

    -- Log completion
    log_message := format('PBatch1.sql: ETPProRq populated with %s records. EndTime: %s, Elapsed: %s', 
                         oid_counter - 1, TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), TO_CHAR(NOW() - start_time, 'MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE 'ETPProRq population completed. Total records added: %', oid_counter - 1;
END $$;

