-- PBatch4.sql
-- Purpose: Waits for ETPProRt to be populated by ETPProt.exe, updates ETPProBt, logs results
-- Version: TriggerBuffered_v251010a, 2025-10-10 (Fixed value too long error by using TIMESTAMP types)
DO $$
DECLARE
    rt_count BIGINT;
    retry_count INTEGER := 0;
    max_retries INTEGER := 60; -- 5 minutes at 5-second intervals
    retry_delay INTEGER := 5; -- seconds
    log_message TEXT;
BEGIN
    -- Verify ETPProRt accessibility
    IF NOT EXISTS (
        SELECT FROM information_schema.tables 
        WHERE table_name = 'etpprort'
    ) THEN
        log_message := 'PBatch4.sql: ETPProRt table does not exist.';
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE EXCEPTION '%', log_message;
    END IF;-- Loop to check ETPProRt
LOOP
    retry_count := retry_count + 1;
    BEGIN
        SELECT COUNT(*) INTO rt_count FROM ETPProRt;
    EXCEPTION
        WHEN OTHERS THEN
            log_message := format('PBatch4.sql: Error querying ETPProRt: %s', SQLERRM);
            INSERT INTO etp_log (log_time, log_message)
            VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
            RAISE EXCEPTION '%', log_message;
    END;
    log_message := format('PBatch4.sql: Retry %s/%s: ETPProRt has %s rows', retry_count, max_retries, rt_count);
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    IF rt_count > 0 THEN
        log_message := format('PBatch4.sql: ETPProRt contains %s rows. Processing complete.', rt_count);
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        -- Embedded PBatch7.sql logic
        DECLARE
            current_batch_oid BIGINT;
            end_time TIMESTAMP := NOW();
            start_time_ts TIMESTAMP;
            execution_time TEXT;
        BEGIN
            SELECT MAX(oid) INTO current_batch_oid FROM ETPProBt WHERE Status = 'Processing';
            IF current_batch_oid IS NULL THEN
                log_message := 'PBatch4.sql: No Processing batch found in ETPProBt.';
                INSERT INTO etp_log (log_time, log_message)
                VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
            ELSE
                SELECT StartTime::TIMESTAMP INTO start_time_ts FROM ETPProBt WHERE oid = current_batch_oid;
                IF start_time_ts IS NULL THEN
                    log_message := format('PBatch4.sql: Start time not found for batch OID: %s', current_batch_oid);
                    INSERT INTO etp_log (log_time, log_message)
                    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
                ELSE
                    execution_time := TO_CHAR(end_time - start_time_ts, 'HH24:MI:SS');
                    UPDATE ETPProBt
                    SET EndTime = TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'),
                        ExecutionTime = execution_time,
                        Status = 'Completed'
                    WHERE oid = current_batch_oid;
                    log_message := format('PBatch4.sql: Batch OID %s marked as Completed. EndTime: %s, ExecutionTime: %s',
                                         current_batch_oid, TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'), execution_time);
                    INSERT INTO etp_log (log_time, log_message)
                    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
                END IF;
            END IF;
        EXCEPTION
            WHEN OTHERS THEN
                log_message := format('PBatch4.sql: Error updating ETPProBt: %s', SQLERRM);
                INSERT INTO etp_log (log_time, log_message)
                VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        END;
        EXIT;
    END IF;
    IF retry_count >= max_retries THEN
        log_message := format('PBatch4.sql: Max retries (%s) reached. ETPProRt still empty.', max_retries);
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE EXCEPTION '%', log_message;
    END IF;
    PERFORM pg_sleep(retry_delay);
END LOOP;
END $$;

