-- PBatch6.sql
-- Purpose: Truncate and repopulate ETPProTp with masking rules based on ETPProHd.DBTableName
-- Version: DynamicReplace_v250827, 2025-08-27
DO $$ 
DECLARE
    column_record RECORD;
    field_number INT := 1;
    mask_type TEXT;
    template_number BIGINT;
    table_name TEXT;
    start_time TIMESTAMP := NOW();
    log_message TEXT;
BEGIN
    -- Log start
    log_message := format('PBatch6.sql: Starting ETPProTp repopulation for dynamic field adjustments. StartTime: %s', 
                         TO_CHAR(start_time, 'YYYY-MM-DD HH24:MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;-- Fetch TemplateNumber and DBTableName directly from ETPProHd
SELECT TemplateNumber, DBTableName 
INTO template_number, table_name 
FROM ETPProHd 
LIMIT 1;

-- Log table selection
log_message := format('PBatch6.sql: Processing table %s with TemplateNumber %s. Elapsed: %s', 
                     table_name, template_number, TO_CHAR(NOW() - start_time, 'MI:SS'));
INSERT INTO etp_log (log_time, log_message)
VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
RAISE NOTICE '%', log_message;

-- Truncate ETPProTp for TemplateNumber
DELETE FROM ETPProTp WHERE TemplateNumber = template_number;
log_message := format('PBatch6.sql: Cleared ETPProTp for TemplateNumber %s. Elapsed: %s', 
                     template_number, TO_CHAR(NOW() - start_time, 'MI:SS'));
INSERT INTO etp_log (log_time, log_message)
VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
RAISE NOTICE '%', log_message;

-- Loop through columns
FOR column_record IN
    EXECUTE format('SELECT column_name, data_type FROM information_schema.columns WHERE table_name = %L', table_name)
LOOP
    -- Assign mask type
    IF column_record.data_type = 'character varying' OR column_record.data_type = 'text' THEN
        mask_type := 'text';
    ELSIF column_record.data_type = 'date' THEN
        mask_type := 'date';
    ELSIF column_record.data_type = 'integer' OR column_record.data_type = 'numeric' THEN
        mask_type := 'number';
    ELSE
        mask_type := 'unknown';
    END IF;

    -- Insert into ETPProTp
    INSERT INTO ETPProTp (TemplateNumber, FieldName, FieldNumber, MaskType)
    VALUES (template_number, column_record.column_name, field_number, mask_type);

    log_message := format('PBatch6.sql: Added %s (FieldNumber: %s, MaskType: %s). Elapsed: %s', 
                         column_record.column_name, field_number, mask_type, TO_CHAR(NOW() - start_time, 'MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    field_number := field_number + 1;
END LOOP;

-- Log completion
log_message := format('PBatch6.sql: ETPProTp repopulated successfully for TemplateNumber %s. EndTime: %s, Elapsed: %s', 
                     template_number, TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), TO_CHAR(NOW() - start_time, 'MI:SS'));
INSERT INTO etp_log (log_time, log_message)
VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
RAISE NOTICE '%', log_message;
END $$;
