-- PBatch7.sql
-- Purpose: Update Completion Time for Latest Batch
-- Version: TriggerBuffered_v251010a, 2025-10-10 (Fixed value too long error by using TIMESTAMP for start_time)
DO $$
DECLARE
    current_batch_oid BIGINT;
    end_time TIMESTAMP := NOW();
    start_time_ts TIMESTAMP;
    execution_time TEXT;
    log_message TEXT;
BEGIN
    -- Log start
    log_message := format('PBatch7.sql: Starting ETPProBt update. StartTime: %s', 
                         TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- Get the most recent batch by OID
    SELECT MAX(oid) INTO current_batch_oid FROM etpprobt;

    IF current_batch_oid IS NULL THEN
        log_message := 'PBatch7.sql: No batch found in ETPProBt.';
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE EXCEPTION '%', log_message;
    END IF;

    -- Fetch its start time as TIMESTAMP
    SELECT StartTime::TIMESTAMP INTO start_time_ts FROM etpprobt WHERE oid = current_batch_oid;

    IF start_time_ts IS NULL THEN
        log_message := format('PBatch7.sql: Start time not found for batch OID: %s', current_batch_oid);
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE EXCEPTION '%', log_message;
    END IF;

    -- Calculate execution duration
    execution_time := TO_CHAR(end_time - start_time_ts, 'HH24:MI:SS');

    -- Update batch record with end time, execution time, and set status to Completed
    UPDATE etpprobt
    SET
        EndTime = TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'),
        ExecutionTime = execution_time,
        Status = 'Completed'
    WHERE oid = current_batch_oid;

    -- Log completion
    log_message := format('PBatch7.sql: Batch OID %s marked as Completed. EndTime: %s, ExecutionTime: %s', 
                         current_batch_oid, TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'), execution_time);
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;
END $$;