-- PBatch789.sql
-- Purpose: Combined PBatch7,8,9: Update batch completion, apply masking to source table, populate ETPProAC
-- Version: v251011a, 2025-10-11 (Merged for efficiency and atomic execution)
DO $$
DECLARE
    current_batch_oid BIGINT;
    end_time TIMESTAMP := NOW();
    start_time_ts TIMESTAMP;
    execution_time TEXT;
    log_message TEXT;
    return_rec RECORD;
    mapping RECORD;
    mask_type TEXT := 'text';
    masked_value TEXT;
    source_id BIGINT;
    unique_id_column TEXT;
    field_length INT;
    processed_count INT := 0;
    inserted_count INT;
    elapsed_time TEXT;
BEGIN
    -- Log start of combined batch
    log_message := format('PBatch789.sql: Starting combined batch 7-9. StartTime: %s', 
                         TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- PBatch7 Logic: Update Completion Time for Latest Batch
    -- Get the most recent batch by OID
    SELECT MAX(oid) INTO current_batch_oid FROM etpprobt;

    IF current_batch_oid IS NULL THEN
        log_message := 'PBatch789.sql (7): No batch found in ETPProBt.';
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE EXCEPTION '%', log_message;
    END IF;

    -- Fetch its start time as TIMESTAMP
    SELECT StartTime::TIMESTAMP INTO start_time_ts FROM etpprobt WHERE oid = current_batch_oid;

    IF start_time_ts IS NULL THEN
        log_message := format('PBatch789.sql (7): Start time not found for batch OID: %s', current_batch_oid);
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE EXCEPTION '%', log_message;
    END IF;

    -- Calculate execution duration
    execution_time := TO_CHAR(end_time - start_time_ts, 'HH24:MI:SS');

    -- Update batch record with end time, execution time, and set status to Completed
    UPDATE etpprobt
    SET
        EndTime = TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'),
        ExecutionTime = execution_time,
        Status = 'Completed'
    WHERE oid = current_batch_oid;

    -- Log PBatch7 completion
    log_message := format('PBatch789.sql (7): Batch OID %s marked as Completed. EndTime: %s, ExecutionTime: %s', 
                         current_batch_oid, TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'), execution_time);
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- PBatch8 Logic: Apply masking rules to sensitive fields in source table
    -- Log start of masking
    log_message := format('PBatch789.sql (8): Starting masking process. StartTime: %s', 
                         TO_CHAR(end_time, 'YYYY-MM-DD HH24:MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- Fetch unique source column from ETPProHd
    SELECT uniquesourcecolumn INTO unique_id_column FROM etpprohd LIMIT 1;
    IF unique_id_column IS NULL THEN
        log_message := 'PBatch789.sql (8): No unique source column found in ETPProHd.';
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE EXCEPTION '%', log_message;
    END IF;
    log_message := format('PBatch789.sql (8): Using unique source column: %s. Elapsed: %s', 
                         unique_id_column, TO_CHAR(NOW() - end_time, 'MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- Process each ETPProRt record
    FOR return_rec IN SELECT * FROM etpprort LOOP
        elapsed_time := TO_CHAR(NOW() - end_time, 'MI:SS');
        log_message := format('PBatch789.sql (8): Processing ImportID: %s. Elapsed: %s', 
                             return_rec.importid, elapsed_time);
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE NOTICE '%', log_message;

        -- Apply masking for each field mapping
        FOR mapping IN
            SELECT * FROM etpprofm
            WHERE templatenumber = return_rec.templatenumber
        LOOP
            -- Get field length from source table
            SELECT character_maximum_length INTO field_length
            FROM information_schema.columns
            WHERE table_name = return_rec.dbtablename AND column_name = mapping.datafieldname;

            -- Get mask type from ETPProTp
            SELECT etp.masktype INTO mask_type 
            FROM etpprotp etp
            WHERE etp.fieldname = mapping.datafieldname;

            IF mask_type IS NULL THEN
                mask_type := 'text';
            END IF;

            -- Apply masking based on type
            IF mask_type = 'text' THEN
                masked_value := repeat('*', GREATEST(field_length - 1, 1));
            ELSIF mask_type = 'date' THEN
                masked_value := '1801-01-01';
            ELSIF mask_type = 'number' THEN
                masked_value := '0';
            ELSE
                masked_value := 'XXXXX';
            END IF;

            log_message := format('PBatch789.sql (8): Applying masked value: %s to field: %s. Elapsed: %s', 
                                 masked_value, mapping.datafieldname, elapsed_time);
            INSERT INTO etp_log (log_time, log_message)
            VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
            RAISE NOTICE '%', log_message;

            -- Update source table with masked value
            EXECUTE format(
                'UPDATE %I SET %I = %L WHERE %I = %L',
                return_rec.dbtablename,
                mapping.datafieldname,
                masked_value,
                unique_id_column,
                return_rec.importid
            );

            log_message := format('PBatch789.sql (8): Updated field %s with masked value %s for ImportID %s. Elapsed: %s', 
                                 mapping.datafieldname, masked_value, return_rec.importid, elapsed_time);
            INSERT INTO etp_log (log_time, log_message)
            VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
            RAISE NOTICE '%', log_message;

            processed_count := processed_count + 1;
        END LOOP;
    END LOOP;

    -- Log PBatch8 completion
    elapsed_time := TO_CHAR(NOW() - end_time, 'MI:SS');
    log_message := format('PBatch789.sql (8): Masking completed successfully for %s records. EndTime: %s, Elapsed: %s', 
                         processed_count, TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), elapsed_time);
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- PBatch9 Logic: Populate ETPProAC from protection return results (ETPProRt)
    -- Log start
    log_message := format('PBatch789.sql (9): Starting ETPProAC population. StartTime: %s', 
                         TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- Insert successful protection returns into ETPProAC
    INSERT INTO etpproac (
        importid, 
        acode, 
        guestbook, 
        templatenumber, 
        username, 
        dbtablename
    )
    SELECT
        r.importid,
        r.acode,
        r.guestbook,
        r.templatenumber,
        r.username,
        (SELECT dbtablename FROM etpprohd LIMIT 1)
    FROM etpprort r
    WHERE r.acode IS NOT NULL;

    -- Get inserted row count
    GET DIAGNOSTICS inserted_count = ROW_COUNT;

    -- Log PBatch9 completion
    log_message := format('PBatch789.sql (9): ETPProAC populated with %s records. EndTime: %s, Elapsed: %s', 
                         inserted_count, TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), 
                         TO_CHAR(NOW() - end_time, 'MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- Log overall completion
    log_message := format('PBatch789.sql: Combined batch 7-9 completed. EndTime: %s', 
                         TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;
END $$;
