-- PBatch8.sql
-- Purpose: Apply masking rules to sensitive fields in source table
-- Version: TriggerBuffered_v250715j, 2025-07-15
DO $$
DECLARE
    return_rec RECORD;
    mapping RECORD;
    mask_type TEXT := 'text';
    masked_value TEXT;
    source_id BIGINT;
    unique_id_column TEXT;
    field_length INT;
    processed_count INT := 0;
    start_time TIMESTAMP := NOW();
    elapsed_time TEXT;
    log_message TEXT;
BEGIN
    -- Log start of masking
    log_message := format('PBatch8.sql: Starting masking process. StartTime: %s', 
                         TO_CHAR(start_time, 'YYYY-MM-DD HH24:MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- Fetch unique source column from ETPProHd
    SELECT uniquesourcecolumn INTO unique_id_column FROM etpprohd LIMIT 1;
    IF unique_id_column IS NULL THEN
        log_message := 'PBatch8.sql: No unique source column found in ETPProHd.';
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE EXCEPTION '%', log_message;
    END IF;
    log_message := format('PBatch8.sql: Using unique source column: %s. Elapsed: %s', 
                         unique_id_column, TO_CHAR(NOW() - start_time, 'MI:SS'));
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;

    -- Process each ETPProRt record
    FOR return_rec IN SELECT * FROM etpprort LOOP
        elapsed_time := TO_CHAR(NOW() - start_time, 'MI:SS');
        log_message := format('PBatch8.sql: Processing ImportID: %s. Elapsed: %s', 
                             return_rec.importid, elapsed_time);
        INSERT INTO etp_log (log_time, log_message)
        VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
        RAISE NOTICE '%', log_message;

        -- Apply masking for each field mapping
        FOR mapping IN
            SELECT * FROM etpprofm
            WHERE templatenumber = return_rec.templatenumber
        LOOP
            -- Get field length from source table
            SELECT character_maximum_length INTO field_length
            FROM information_schema.columns
            WHERE table_name = return_rec.dbtablename AND column_name = mapping.datafieldname;

            -- Get mask type from ETPProTp
            SELECT etp.masktype INTO mask_type 
            FROM etpprotp etp
            WHERE etp.fieldname = mapping.datafieldname;

            IF mask_type IS NULL THEN
                mask_type := 'text';
            END IF;

            -- Apply masking based on type
            IF mask_type = 'text' THEN
                masked_value := repeat('*', GREATEST(field_length - 1, 1));
            ELSIF mask_type = 'date' THEN
                masked_value := '1801-01-01';
            ELSIF mask_type = 'number' THEN
                masked_value := '0';
            ELSE
                masked_value := 'XXXXX';
            END IF;

            log_message := format('PBatch8.sql: Applying masked value: %s to field: %s. Elapsed: %s', 
                                 masked_value, mapping.datafieldname, elapsed_time);
            INSERT INTO etp_log (log_time, log_message)
            VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
            RAISE NOTICE '%', log_message;

            -- Update source table with masked value
            EXECUTE format(
                'UPDATE %I SET %I = %L WHERE %I = %L',
                return_rec.dbtablename,
                mapping.datafieldname,
                masked_value,
                unique_id_column,
                return_rec.importid
            );

            log_message := format('PBatch8.sql: Updated field %s with masked value %s for ImportID %s. Elapsed: %s', 
                                 mapping.datafieldname, masked_value, return_rec.importid, elapsed_time);
            INSERT INTO etp_log (log_time, log_message)
            VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
            RAISE NOTICE '%', log_message;

            processed_count := processed_count + 1;
        END LOOP;
    END LOOP;

    -- Log completion
    elapsed_time := TO_CHAR(NOW() - start_time, 'MI:SS');
    log_message := format('PBatch8.sql: Masking completed successfully for %s records. EndTime: %s, Elapsed: %s', 
                         processed_count, TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), elapsed_time);
    INSERT INTO etp_log (log_time, log_message)
    VALUES (TO_CHAR(NOW(), 'YYYY-MM-DD HH24:MI:SS'), log_message);
    RAISE NOTICE '%', log_message;
END $$;