-- RunTriggeredPipeline.sql (v251007a)
-- Purpose: Atomic pipeline: Flush buffer to PRQ, claim 'Ready' batch to 'Processing', process sequence, finalize
-- Called from Clarion with batch handling in SQL for no race

DO $$
DECLARE
    dyn_table TEXT;
    flushed_count INT := 0;
    claimed_batchnumber BIGINT;
    start_time TIMESTAMP := CURRENT_TIMESTAMP;
BEGIN
    -- Log start
    INSERT INTO etp_log (log_time, log_message) 
    VALUES (TO_CHAR(NOW(),'YYYY-MM-DD HH24:MI:SS'), 'RunTriggeredPipeline: Starting pipeline.');

    -- Get dynamic table and values
    SELECT DBTableName INTO dyn_table FROM ETPProHd LIMIT 1;

    -- Truncate PRQ if buffer has data
    IF EXISTS (SELECT 1 FROM etpprobf) THEN
        TRUNCATE TABLE etpproRq;
        RAISE NOTICE 'PRQ truncated before flush.';
    END IF;

    -- Flush from buffer to PRQ (your original loop as bulk INSERT for efficiency)
    INSERT INTO etpproRq (
        oid, guestbook, location, templatenumber, username, dbtablename, importid, datafieldnumber, datafieldname, datavalue
    ) SELECT
        row_number() OVER () AS oid,  -- Assign new OIDs
        guestbook, location, templatenumber, username, dbtablename, importid, datafieldnumber, datafieldname, datavalue
    FROM etpprobf ORDER BY importid, datafieldnumber;

    GET DIAGNOSTICS flushed_count = ROW_COUNT;
    RAISE NOTICE 'Flushed % records to PRQ.', flushed_count;

    -- Clear buffer if flushed
    IF flushed_count > 0 THEN
        DELETE FROM etpprobf;
    END IF;

    -- Create 'Ready' batch if flushed (your original logic)
    IF flushed_count > 0 THEN
        INSERT INTO etpprobt (
            BatchNumber, TemplateNumber, UserName, DBTableName, RecordCount, StartTime, EndTime, ExecutionTime, Status
        ) SELECT
            COALESCE(MAX(BatchNumber), 0) + 1, TemplateNumber, UserName, DBTableName, COUNT(DISTINCT importid),
            start_time, CURRENT_TIMESTAMP, TO_CHAR(CURRENT_TIMESTAMP - start_time, 'HH24:MI:SS'), 'Ready'
        FROM ETPProHd, etpproRq GROUP BY TemplateNumber, UserName, DBTableName;
    ELSE
        RAISE NOTICE 'No records flushed—no batch created.';
        RETURN;
    END IF;

    -- Claim 'Ready' to 'Processing' (atomic)
    WITH next_batch AS (
        SELECT BatchNumber FROM etpprobt WHERE Status = 'Ready' ORDER BY BatchNumber DESC LIMIT 1
    ) UPDATE etpprobt
    SET Status = 'Processing'
    WHERE BatchNumber IN (SELECT BatchNumber FROM next_batch)
    RETURNING BatchNumber INTO claimed_batchnumber;

    IF claimed_batchnumber IS NULL THEN
        RAISE NOTICE 'No Ready batch to claim—exiting.';
        RETURN;
    END IF;

    -- Log claim
    INSERT INTO etp_log (log_time, log_message) 
    VALUES (TO_CHAR(NOW(),'YYYY-MM-DD HH24:MI:SS'), 'Pipeline: Claimed batch ' || claimed_batchnumber || ' as Processing.');

    -- Run pipeline sequence (inline or call functions; adjust to your PBatch3-9 logic)
    -- Example for PBatch3: Process PRQ to Rt (replace with your actual SQL)
    -- INSERT INTO ETPProRt SELECT * FROM ETPProRq WHERE batchnumber = claimed_batchnumber;  -- Placeholder
    -- Similarly for 4-9...

    -- Log completion
    INSERT INTO etp_log (log_time, log_message) 
    VALUES (TO_CHAR(NOW(),'YYYY-MM-DD HH24:MI:SS'), 'Pipeline: Completed batch ' || claimed_batchnumber || '.');
END $$;