-- TrigPipeline.sql (v251011a)
-- Purpose: Atomic pipeline: Flush buffer to PRQ, find existing 'Ready' PBT, update that PBT with flushed_count as RecordCount and mark as 'Processing' after flush (no new batch creation)
-- Called from Clarion with batch handling in SQL for no race

DO $$
DECLARE
    dyn_table TEXT;
    flushed_count INT := 0;
    ready_batchnumber BIGINT;
    start_time TIMESTAMP := CURRENT_TIMESTAMP;
BEGIN
    -- Log start
    INSERT INTO etp_log (log_time, log_message) 
    VALUES (TO_CHAR(NOW(),'YYYY-MM-DD HH24:MI:SS'), 'TrigPipeline: Starting pipeline.');

    -- Get dynamic table and values
    SELECT DBTableName INTO dyn_table FROM ETPProHd LIMIT 1;

    -- Truncate PRQ if buffer has data
    IF EXISTS (SELECT 1 FROM etpprobf) THEN
        TRUNCATE TABLE etpproRq;
        RAISE NOTICE 'PRQ truncated before flush.';
    END IF;

    -- Flush from buffer to PRQ (bulk INSERT for efficiency)
    INSERT INTO etpproRq (
        oid, guestbook, location, templatenumber, username, dbtablename, importid, datafieldnumber, datafieldname, datavalue
    ) SELECT
        row_number() OVER () AS oid,  -- Assign new OIDs
        guestbook, location, templatenumber, username, dbtablename, importid, datafieldnumber, datafieldname, datavalue
    FROM etpprobf ORDER BY importid, datafieldnumber;

    GET DIAGNOSTICS flushed_count = ROW_COUNT;
    RAISE NOTICE 'Flushed % records to PRQ.', flushed_count;

    -- Clear buffer if flushed
    IF flushed_count > 0 THEN
        DELETE FROM etpprobf;
    END IF;

    IF flushed_count = 0 THEN
        RAISE NOTICE 'No records flushed—exiting.';
        RETURN;
    END IF;

    -- Find the existing latest 'Ready' PBT (created by insert trigger)
    SELECT BatchNumber INTO ready_batchnumber 
    FROM etpprobt 
    WHERE Status = 'Ready' 
    ORDER BY BatchNumber DESC LIMIT 1;

    IF ready_batchnumber IS NULL THEN
        RAISE NOTICE 'No Ready batch found—exiting.';
        INSERT INTO etp_log (log_time, log_message) 
        VALUES (TO_CHAR(NOW(),'YYYY-MM-DD HH24:MI:SS'), 'TrigPipeline: No Ready batch after flush.');
        RETURN;
    END IF;

    -- Log found Ready batch
    INSERT INTO etp_log (log_time, log_message) 
    VALUES (TO_CHAR(NOW(),'YYYY-MM-DD HH24:MI:SS'), 'TrigPipeline: Found Ready batch ' || ready_batchnumber || '.');

    -- Update the existing PBT with flushed_count as RecordCount, timestamps, and set to 'Processing' (single UPDATE after flush)
    UPDATE etpprobt
    SET 
        RecordCount = flushed_count,
        StartTime = start_time,
        EndTime = CURRENT_TIMESTAMP,
        ExecutionTime = TO_CHAR(CURRENT_TIMESTAMP - start_time, 'HH24:MI:SS'),
        Status = 'Processing'
    WHERE BatchNumber = ready_batchnumber;

    -- Log update to Processing
    INSERT INTO etp_log (log_time, log_message) 
    VALUES (TO_CHAR(NOW(),'YYYY-MM-DD HH24:MI:SS'), 'TrigPipeline: Updated batch ' || ready_batchnumber || ' to Processing with RecordCount ' || flushed_count || '.');
END $$;
